-- Parst das erste EDI-Segment, welches die Trennungszeichen enthält
-- Das Segment sieht üblicherweise so aus: UNA:+.? '
CREATE OR REPLACE FUNCTION tedifact.parseUNA( _edifact varchar ) RETURNS tedifact.service_string_advice AS $$
DECLARE
  _una varchar;
  _ssa tedifact.service_string_advice;
BEGIN

  _una := substring( _edifact FROM 1 FOR 9 ); -- Segment besteht aus genau 9 Zeichen
  IF 
       length( _una ) <> 9 
    OR _una NOT LIKE 'UNA%' -- Segment heißt UNA
    OR substring( _una FROM 8 FOR 1 ) <> ' ' -- Das vorletzte Zeichen ist stets ein Leerzeichen
  THEN
    RAISE EXCEPTION 'Ungültiges UNA-Segment >%<', _una;
  END IF;
  
  _ssa.component_data_element_separator := substring( _una FROM 4 FOR 1 ); -- Trenner der Datenelemente - üblicherweise ein :
  _ssa.data_element_separator := substring( _una FROM 5 FOR 1 ); -- Trenner der Subelemente - üblicherweise ein +
  _ssa.decimal_notation := substring( _una FROM 6 FOR 1 ); -- Dezimaltrenner - üblicherweise ein -
  _ssa.release_indicator := substring( _una FROM 7 FOR 1 ); -- Escapecharakter - üblicherweise ein ?
  _ssa.segment_terminator := substring( _una FROM 9 FOR 1 ); -- Segmenttrenner - üblicherweise ein '
  
  RETURN _ssa;

END $$ LANGUAGE plpgsql STRICT IMMUTABLE;
--
       
     
------------------------------------------------------------------------------------------
-- Einlesen von EDI-Spezifikationen von https://www.stylusstudio.com/edifact/frames.htm --
------------------------------------------------------------------------------------------

-- Lädt die Spezifikation eines Subsegments
-- Beispiel des Originals:  020     C110  PAYMENT TERMS   C   1
-- Reihenfolge: Position Code Bezeichnung Mandatoryflag Anzahl
CREATE OR REPLACE FUNCTION tedifact.subsegment__descr__add( _release_type varchar, _release_version varchar, _segment varchar, _descr varchar, _separator varchar = '$t$' ) RETURNS integer AS $$
DECLARE
  _fields varchar[];
  _segpos integer;
  _segcode varchar;
  _segbez varchar;
  _segman boolean;
  _segrep integer;
  _eseg_id integer;
  _format varchar;
  _edat_alpha boolean;
  _edat_numeric boolean;
  _edat_length integer;
  _edat_length_exact boolean;  
BEGIN

  _fields := string_to_array( _descr, _separator ); -- Zerlegung in Felder
  _segpos := trim( _fields[1] )::integer; 
  _segcode := trim( _fields[3] );
  _segbez := trim( _fields[4] );
  IF trim( _fields[5] ) = 'M' THEN -- 'M' heißt Pflichtfeld, 'C' optional
    _segman := true;
  ELSE
    _segman := false;  
  END IF;
  _segrep := trim( _fields[6] )::integer; -- Wie oft kann das Subsegment innerhalb des Segments auftreten?
  
  INSERT INTO tedifact.subsegments
    ( subs_segment, subs_release_type, subs_release_version, subs_pos, subs_code, subs_bez, subs_mandatory, subs_repeat, subs_insert_date )
  VALUES
    ( _segment,  _release_type, _release_version, _segpos, _segcode, _segbez, _segman, _segrep, now() )
  RETURNING subs_id INTO _eseg_id;
  
  -- Eine Segmentspezifikation enthält normalerweise keine Formatinfos, das Folgende dient nur zur Sicherheit
  _format := _fields[7];
  IF coalesce( _format, '' ) <> '' THEN
  
    SELECT * INTO _edat_alpha, _edat_numeric, _edat_length, _edat_length_exact FROM tedifact.representation__parse( _format );    
    INSERT INTO tedifact.dataelements
      ( data_subs_id, data_pos, data_code, data_bez, data_mandatory, data_alpha, data_numeric, data_length, data_length_exact, data_insert_date )
    VALUES
      ( _eseg_id, 1, _segcode, _segbez, _segman, _edat_alpha, _edat_numeric, _edat_length, _edat_length_exact, now() ); 
  
  END IF;
  
  RETURN _eseg_id; 

END $$ LANGUAGE plpgsql STRICT;
--

-- parst Formatinformationen 
-- Beispiel: an..1024
-- a alphanumerisch, n numerisch
-- Zahl gibt Länge des Datenelements an
-- mit .. Länge = Maximallänge, sonst Länge 0 exakte Länge
CREATE OR REPLACE FUNCTION tedifact.representation__parse( _repr varchar, OUT alpha boolean, OUT num boolean, OUT len integer, OUT len_exact boolean )
RETURNS record AS $$ 
BEGIN

  alpha := _repr LIKE 'a%'; 
  num := _repr LIKE '%n%';
  len := regexp_replace( _repr, '[^0123456789]', '', 'g' )::integer;
  len_exact := _repr NOT LIKE '%..%';
  
END $$ LANGUAGE plpgsql STRICT;
--

-- Liest die Spezifikationen eines Datenelements ein
-- Beispiel des Originals:  020     C110  PAYMENT TERMS   C   1
-- Vorgehen analog zum Einlesen der Subsegmentspezifikationen (tedifact.subsegment__descr__add),
--   nur dass es hier Formatinformationen, aber keine Positionsangabe gibt 
CREATE OR REPLACE FUNCTION tedifact.dataelement__descr__add( _eseg_id integer, _pos integer, _descr varchar, _separator varchar = '\t' ) RETURNS void AS $$
DECLARE
  _fields varchar[];
  _code varchar;
  _bez varchar;
  _man boolean;
  _format varchar;
  _edat_alpha boolean;
  _edat_numeric boolean;
  _edat_length integer;
  _edat_length_exact boolean;
BEGIN

  _fields := string_to_array( _descr, _separator );
  _code := trim( _fields[3] );
  _bez := trim( _fields[4] );
  IF trim( _fields[5] ) = 'M' THEN
    _man := true;
  ELSE
    _man := false;  
  END IF; 
  
  _format := trim( _fields[7] );
  SELECT * INTO _edat_alpha, _edat_numeric, _edat_length, _edat_length_exact FROM tedifact.representation__parse( _format );
  
  INSERT INTO tedifact.dataelements
    ( data_subs_id, data_pos, data_code, data_bez, data_mandatory, data_alpha, data_numeric, data_length, data_length_exact, data_insert_date )
  VALUES
    ( _eseg_id, _pos, _code, _bez, _man, _edat_alpha, _edat_numeric, _edat_length, _edat_length_exact, now() ); 

END $$ LANGUAGE plpgsql STRICT;
--

-- Liest die Spezifikationen aller Subsegmente und Datenelemente eines EDI-Segments ein
-- Beispiel:
-- 010     7077  Description format code   C   1   an..3   
-- 020     C272  ITEM CHARACTERISTIC   C   1     
--         7081  Item characteristic code  C     an..3   
--         1131  Code list identification code   C     an..17  
--         3055  Code list responsible agency code   C     an..3   
-- 030     C273  ITEM DESCRIPTION  C   1     
--         7009  Item description code   C     an..17  
--         1131  Code list identification code   C     an..17  
--         3055  Code list responsible agency code   C     an..3   
--         7008  Item description  C     an..256   
--         7008  Item description  C     an..256   
--         3453  Language name code  C     an..3   
-- 040     7383  Surface or layer code   C   1   an..3   
CREATE OR REPLACE FUNCTION tedifact.segment__descr__add( _release_type varchar, _release_version varchar, _segment varchar, _descr varchar, _separator_n varchar = E'\n', _separator_t varchar = E'\t' ) RETURNS void AS $$
DECLARE
  _lines varchar[];
  _line varchar;
  _i integer;
  _eseg_id integer;
  _pos integer;
BEGIN

  _lines := string_to_array( _descr, _separator_n );
  
  FOR _i IN 1..array_length( _lines, 1 ) LOOP
    
    _line := _lines[_i];
    IF starts_with( _line, _separator_t ) THEN
      PERFORM tedifact.dataelement__descr__add( _eseg_id, _pos, _line, _separator_t );
      _pos := _pos + 1;
    ELSE
      _eseg_id := tedifact.subsegment__descr__add( _release_type, _release_version, _segment, _line, _separator_t );
      _pos := 1;
    END IF; 
    
  END LOOP;

END $$ LANGUAGE plpgsql STRICT;
--

-- Liest eine Zeile der Spezifikation eines Nachrichtentyps ein.
-- Die Spezifikation ist baumartig aufgebaut. Eine Zeile dieser Spezifikation kann ein Segment enthalten
-- (Beispiel │─├─RFF   ×1   (M))
-- oder es sind nur abstrakte Gruppenknoten, welche mehrere Segmente umfassen
-- (Beispiel ├─Group 3   ×99  (C))
CREATE OR REPLACE FUNCTION tedifact.message__descr__line__add( _release_version varchar, _message_type varchar, _descr varchar ) RETURNS void AS $$
DECLARE
  _d varchar;
  _fields varchar[];
  _content_segment boolean;
  _group integer;
  _group_level integer;
  _group_level_temp integer;
  _supergroup integer;
  _segment varchar;
  _repeat integer;
  _mandatory boolean;
BEGIN

  -- HTML-Zeichen in löschen bzw. durch lesbare Zeichen ersetzen
  _d := regexp_replace( _descr, '│', '', 'g' );
  _d := regexp_replace( _d, '├', '', 'g' );
  _d := regexp_replace( _d, '└', '', 'g' );
  _d := regexp_replace( _d, '─', '─ ', 'g' );
  _d := regexp_replace( _d, ' ─', '─', 'g' );
  _d := regexp_replace( _d, 'Group ', 'Group', 'g' );
  _d := regexp_replace( _d, '   ', ' ', 'g' );
  _d := regexp_replace( _d, '  ', ' ', 'g' );
  _d := regexp_replace( _d, '×', '', 'g' );
  _fields := string_to_array( _d, ' ' );
  
  -- Segment obligatorisch/optional
  _repeat := _fields[3]::integer;
  IF _fields[4] LIKE '%(C)%' THEN _mandatory := false; END IF;
  IF _fields[4] LIKE '%(M)%' THEN _mandatory := true; END IF;  
  
  -- Gruppensegmente enthalten nur Metainformation, dafür einen Verweis auf die übergeordnete Gruppe
  IF _fields[2] LIKE 'Group%' THEN
  
    _content_segment := false; -- kein eigentlicher Inhalt
    _group_level := length( _fields[1] ); -- Level = Anzahl der Einrückungen am Anfang
    _group := regexp_replace( _fields[2], 'Group', '' )::integer; -- Gruppenindex herauslösen
    _supergroup := coalesce( (SELECT segm_group FROM tedifact.segments WHERE segm_group_level = _group_level - 1 ORDER BY segm_id DESC LIMIT 1), 0 );
      -- Das Supersegments ist das letzte eingelesene des nächsthöheren Levels.
    _segment := null; 
  
  ELSE
   
    _content_segment := true; -- enthält ein richtiges Segments
    SELECT segm_group, segm_group_level INTO _group, _group_level_temp FROM tedifact.segments WHERE NOT segm_content_segment ORDER BY segm_id DESC LIMIT 1;
    _group := coalesce( _group, 0 );
    _group_level := length( _fields[1] );

    -- Fallunterscheidung für Segmente des höchsten Levels
    IF _group_level_temp < _group_level THEN
      _group_level := _group_level_temp + 1;
    ELSE
      _group_level := 1;
      _group := 0;
    END IF;
    _supergroup := null; -- Hier kein Verweis auf Supergruppe, den enthält der zugehörige Gruppendatensatz.
    _segment := substring( _fields[2] FROM 1 FOR 3 ); -- Dafür gibt es hier ein Segment.
  
  END IF;
  
  INSERT INTO tedifact.segments
    ( segm_release_version, segm_message_type, segm_content_segment, segm_group, segm_group_level, segm_supergroup, segm_segment, segm_repeat, segm_mandatory, segm_insert_date )
  VALUES
    ( _release_version, _message_type, _content_segment, _group, _group_level, _supergroup, _segment, _repeat, _mandatory, now() ); 

END $$ LANGUAGE plpgsql STRICT;
--


-- Liest eine Spezifikation eines Nachrichtentyps ein.
-- Beispiel (Auszug):
-- UN/CEFACT Release D04B Message ORDRSP 
-- ├─UNH  ×1   (M)
-- ├─BGM  ×1   (M)
-- ├─DTM  ×35  (M)
-- ├─PAI  ×1   (C)
-- ├─ALI  ×5   (C)
-- ├─IMD  ×999   (C)
-- ├─FTX  ×99  (C)
-- ├─GIR  ×10  (C)
-- ├─Group 1  ×9999  (C)
-- │─├─RFF  ×1   (M)
-- │─└─DTM  ×5   (C)
-- ├─Group 2  ×1   (C)
-- │─├─AJT  ×1   (M)
-- │─└─FTX  ×5   (C)
-- ├─Group 3  ×99  (C)
-- │─├─NAD  ×1   (M)
-- │─├─LOC  ×25  (C)
-- │─├─FII  ×5   (C)
-- │─├─Group 4  ×99  (C)
-- │─│─├─RFF  ×1   (M)
-- ...
CREATE OR REPLACE FUNCTION tedifact.message__descr__add( _descr varchar, _separator_n varchar = E'\n' ) RETURNS void AS $$
DECLARE
  _lines varchar[];
  _head varchar;
  _release_version varchar; 
  _message_type varchar;
  _i integer;
  _line varchar;
BEGIN

  _lines := string_to_array( _descr, _separator_n ); -- Text anhand der Zeilenumbrüche zerlegen

  -- Headerinfos auslesen
  _head := _lines[1];
  _release_version := split_part( _head, ' ', 3 );
  _message_type := split_part( _head, ' ', 5 );
  
  -- Gibt es noch nichts dazu in der DB, dann Wurzelsegment hinterlegen
  IF NOT EXISTS( SELECT 1 FROM tedifact.segments WHERE segm_release_version = _release_version AND segm_message_type = _message_type ) THEN
  
    INSERT INTO tedifact.segments
      ( segm_release_version, segm_message_type, segm_content_segment, segm_group, segm_group_level, segm_supergroup, segm_segment, segm_repeat, segm_mandatory, segm_insert_date )
    VALUES
      ( _release_version, _message_type, false, 0, 1, null, null, 1, true, now() );  
  
  END IF;
  
  -- Spezifikation zeilenweise einlesen, Headerzeile übergehen
  FOR _i IN 2..array_length( _lines, 1 ) LOOP
  
    _line := _lines[_i];
    PERFORM tedifact.message__descr__line__add( _release_version, _message_type, _line );
     
  END LOOP;
  
  EXCEPTION WHEN OTHERS THEN
    raise notice '%', SQLERRM;
    raise notice '%', _line;
  
END $$ LANGUAGE plpgsql STRICT;
--


-- Ermittelt alle Segmente eines Nachrichtentyps, zu denen die Spezifikationen für die Subsegmente und Datenelemente felhen,
CREATE OR REPLACE FUNCTION tedifact.segment__descr__missing__find( _release_version varchar, _message_type varchar ) RETURNS SETOF varchar AS $$

  SELECT DISTINCT segm_segment FROM tedifact.segments WHERE 
        segm_release_version = _release_version 
    AND segm_message_type = _message_type
    AND segm_segment IS NOT null
    AND NOT EXISTS ( 
      SELECT 1 FROM tedifact.subsegments
      WHERE  
            subs_release_type = 'CEFACT'
        AND subs_release_version = _release_version
        AND subs_segment = segm_segment
    )
  ORDER BY segm_segment;

$$ LANGUAGE sql STRICT STABLE;
--

-- Wandelt das Mandatoryflag der Spezifiktion in einen String um.
CREATE OR REPLACE FUNCTION tedifact.mandatory__ident( _mand boolean ) RETURNS varchar AS $$

  SELECT ifthen( _mand, '(M)', '(C)' );

$$ LANGUAGE sql IMMUTABLE STRICT;
--


-- Wandelt die Spezifikation eines Segments in einen String um.
CREATE OR REPLACE FUNCTION tedifact.segment__ident( _es_id integer ) RETURNS varchar AS $$

  SELECT coalesce( segm_segment, 'Gruppe ' || segm_group ) || ' ' || tedifact.mandatory__ident( segm_mandatory )
  FROM tedifact.segments
  WHERE segm_id = _es_id; 

$$ LANGUAGE sql STABLE STRICT;
--


-- Wandelt die Spezifikation eines Subsegments in einen String um.
CREATE OR REPLACE FUNCTION tedifact.subsegment__ident( _es_id integer, _eseg_id integer ) RETURNS varchar AS $$

  SELECT coalesce( tedifact.segment__ident( _es_id ) || ' - ', '' ) || subs_code || '/' || subs_pos || '/' || subs_bez
    || '/' || tedifact.mandatory__ident( subs_mandatory )
  FROM tedifact.subsegments
  WHERE subs_id = _eseg_id; 

$$ LANGUAGE sql STABLE;
--

-- Wandelt die Spezifikation eines Datenelements in einen String um.
CREATE OR REPLACE FUNCTION tedifact.dataelement__ident( _es_id integer, _edat_id integer ) RETURNS varchar AS $$

  SELECT coalesce( tedifact.subsegment__ident( _es_id, data_subs_id ) || ' - ', '' ) || data_code 
    || '/' || data_pos || '/' || data_bez || tedifact.mandatory__ident( data_mandatory )
  FROM tedifact.dataelements
  WHERE data_id = _edat_id; 

$$ LANGUAGE sql STABLE STRICT;
--



----------------------------------
-- Einlesen einer EDI-Nachricht --
----------------------------------

-- Bestimmt die passende vorhandene EDI-Version zur EDI-Version der Nachricht
CREATE OR REPLACE FUNCTION tedifact.edi__version__for__parsing__get( _message_edi_version varchar ) RETURNS varchar AS $$

  -- alle vorhandene EDI-Versionen auslesen und sortieren
  WITH available_versions AS (
    SELECT 
      DISTINCT segm_release_version AS edi_version
    FROM tedifact.segments 
    WHERE segm_message_type = 'ORDRSP'
    ORDER BY segm_release_version
  )

  SELECT edi_version FROM (

    -- Aus dem Pool aller vorhandenne EDI-Versionen, die mindestens so groß 
    -- wie die der Nachricht sind...
    SELECT
      edi_version
    FROM available_versions
    WHERE _message_edi_version <= edi_version

    UNION 

    -- ...und der höchsten vorhandenen EDI-Version
    SELECT 
      max( edi_version )
    FROM available_versions

  ) AS x

  -- wird die kleinste ausgewählt.
  ORDER BY edi_version
  LIMIT 1;    

$$ LANGUAGE sql STRICT STABLE;
--

-- Parst die Datenelemente eines Subsegments
CREATE OR REPLACE FUNCTION tedifact.dataelements__parse( _subsegment_id integer, _separators tedifact.service_string_advice ) RETURNS void AS $$
DECLARE
  _dummy varchar = 'gfhuafhqwegfh'; -- Dummy, um maskierte Trennzeichen vorübergehend zu ersetzen
  _no_break varchar;
  _subsegment varchar;
  _edi_id integer;
  _eseg_id integer;
  _dataelements varchar[];
  _dataelement varchar;
  _counter integer;
  _r record;
BEGIN

  -- Klimmzüge wegen maskierter Zeichen
  _no_break := _separators.release_indicator || _separators.component_data_element_separator;

  -- Daten des Subsegments auslesen
  SELECT psubs_inc_id, psubs_content, psubs_subs_id INTO _edi_id, _subsegment, _eseg_id FROM tedifact.edi_parsed_subsegments 
    WHERE psubs_id = _subsegment_id;
  _subsegment := replace( _subsegment, _no_break, _dummy );
  
  -- Datenelemente auseinandernehmen
  _dataelements := string_to_array( _subsegment, _separators.component_data_element_separator );
  _counter := 1;  
  
  -- Alle Datenelemente nacheinander parsen
  FOR _r IN ( SELECT data_id FROM tedifact.dataelements WHERE data_subs_id = _eseg_id ORDER BY data_pos ) LOOP
  
    EXIT WHEN array_length( _dataelements, 1 ) < _counter;
    _dataelement := coalesce( _dataelements[_counter], '' );
    _dataelement := replace( _dataelement, _dummy, _no_break ); 
  
    INSERT INTO tedifact.edi_parsed_dataelements
    ( pdata_inc_id, pdata_psubs_id, pdata_data_id, pdata_content )
  VALUES
    ( _edi_id, _subsegment_id, _r.data_id, _dataelement );
    
  _counter := _counter + 1;  
  
  END LOOP;

END $$ LANGUAGE plpgsql STRICT;
--


-- Parst die Datenelemente und Subsegmente eines Segments
CREATE OR REPLACE FUNCTION tedifact.subsegments__parse( _segment_id integer, _separators tedifact.service_string_advice ) RETURNS void AS $$
DECLARE
  _dummy varchar = 'gfhuafhqwegfh'; -- Dummy, um maskierte Trennzeichen vorübergehend zu ersetzen
  _no_break varchar;
  _segment varchar;
  _edi_id integer;
  _segment_name varchar;
  _subsegments varchar[];
  _subsegment varchar;
  _counter integer;
  _r record;
  _ediss_id integer;
BEGIN

  -- Klimmzüge wegen maskierter Zeichen
  _no_break := _separators.release_indicator || _separators.data_element_separator;
  SELECT psegm_content, psegm_inc_id INTO _segment, _edi_id FROM tedifact.edi_parsed_segments WHERE psegm_id = _segment_id;
  _segment := replace( _segment, _no_break, _dummy );
  
  -- Subsegmente auseinandernehmen
  _subsegments := string_to_array( _segment, _separators.data_element_separator );
  _segment_name := _subsegments[1];
  _counter := 2;
  
  -- Alle Subsegmente nacheinander parsen
  FOR _r IN ( 
    SELECT subs_id 
    FROM tedifact.subsegments 
    WHERE subs_segment = _segment_name 
    ORDER BY subs_pos 
  ) LOOP
  
    -- Sollte wir weniger Subsegmente haben, als die Spezifikation hergibt, dann Ausstieg aus der Schleife.
    EXIT WHEN array_length( _subsegments, 1 ) < _counter;
    _subsegment := _subsegments[_counter];
    _subsegment := replace( _subsegment, _dummy, _no_break );
    
    INSERT INTO tedifact.edi_parsed_subsegments
      ( psubs_inc_id, psubs_psegm_id, psubs_subs_id, psubs_content )
    VALUES
      ( _edi_id, _segment_id, _r.subs_id, _subsegment )
    RETURNING psubs_id INTO _ediss_id;
    
    -- Datenelemente des Subsegments einlesen.
    PERFORM tedifact.dataelements__parse( _ediss_id, _separators );
    
    _counter := _counter + 1;
  
  END LOOP;

END $$ LANGUAGE plpgsql STRICT;
--


-- Parst eine EDI-Nachricht (innere Funktion)
CREATE OR REPLACE FUNCTION tedifact.segments__parse( 
  _edi_id integer, 
  _separators tedifact.service_string_advice 
) RETURNS void AS $$
DECLARE
  _dummy varchar = 'gfhuafhqwegfh'; -- Dummy, um maskierte Trennzeichen vorübergehend zu ersetzen
  _message varchar;
  _message_type varchar;
  _edi_version varchar;
  _no_break varchar;
  _msg varchar;
  _r record;
  _s varchar;
  _seg varchar;
  _seg_id integer;
  _edis_id integer;
BEGIN 

  SELECT inc_content, inc_message_type, inc_edi_version_parsed 
  INTO _message, _message_type, _edi_version FROM tedifact.edi_incoming WHERE inc_id = _edi_id;

  -- Klimmzüge wegen maskierter Zeichen
  _no_break := _separators.release_indicator || _separators.segment_terminator;

  -- Nachricht in Zeilen - Segmente - zerlegen
  _msg := replace( _message, _no_break, _dummy );
  
  -- Segmente zeilenweise parsen
  FOR _r IN ( SELECT regexp_split_to_table( _msg, _separators.segment_terminator ) AS s ) LOOP  
  
    _s := replace( _r.s, _dummy, _separators.segment_terminator );
    --Leerzeilen werden übersprungen
    CONTINUE WHEN coalesce( _s, '' ) = ''; 

    -- Seltsame Zeichen am Zeilenanfang werden ignoriert.
    IF substring( _s, 1, 1 ) !~ '[ABCDEFGHIJKLMNOPQRSTUVWXYZ]' THEN
      _s := substring( _s, 2 );
    END IF; 

    -- Segment-ID steht stets am Anfang und besteht aus drei Großbuchstaben.
    _seg := substring( _s, 1, 3 );

    -- Das Segment UNA ist ein Sonderfall und wird hier übergangen.
    CONTINUE WHEN _seg = 'UNA';

    -- Segmentbeschreibung der passenden EDI-Version auslesen
    _seg_id := min( segm_id ) FROM tedifact.segments 
         WHERE segm_message_type::varchar IN ( _message_type, 'EDIFACT' )
         AND ( segm_release_version = _edi_version OR segm_message_type = 'EDIFACT' )
         AND segm_segment = _seg;

    -- Segment unbekannt? Dann Fehler, ggf. muss eine andere Spezifikation benutzt werden.
    IF _seg_id IS null THEN
      RAISE EXCEPTION 'Unbekanntes Segment % in EDI-Version %', _seg, _edi_version;
    END IF;          

    INSERT INTO tedifact.edi_parsed_segments
      ( psegm_inc_id, psegm_segm_id, psegm_content )
    VALUES
      ( _edi_id, _seg_id, _s )
    RETURNING psegm_id INTO _edis_id;
    
    -- Subsegmente und Datenelemente parsen
    PERFORM tedifact.subsegments__parse( _edis_id, _separators ); 
  
  END LOOP;
  
END $$ LANGUAGE plpgsql STRICT;
--


-- Parst eine EDI-Nachricht (äußere Funktion)
CREATE OR REPLACE FUNCTION tedifact.message__parse( _message_id varchar ) RETURNS void AS $$
DECLARE
  _edi_id integer;
  _message varchar;
  _message_type varchar;
  _separators tedifact.service_string_advice;
BEGIN

  -- EDI-Nachricht aus Importtabelle lesen.
  SELECT inc_id, inc_content, inc_message_type INTO _edi_id, _message, _message_type 
  FROM tedifact.edi_incoming
  WHERE inc_origin = _message_id;
  
  IF _edi_id IS null THEN
    RETURN;
  END IF;
  
  -- Vorherige Einträge zu dieser EDI-Nachricht löschen
  DELETE FROM tedifact.edi_parsed_dataelements WHERE pdata_inc_id = _edi_id;
  DELETE FROM tedifact.edi_parsed_subsegments WHERE psubs_inc_id = _edi_id;
  DELETE FROM tedifact.edi_parsed_segments WHERE psegm_inc_id = _edi_id;
  
  -- UNA-Segment auslesen
  _separators := tedifact.parseUNA( _message );

  -- Restliche Nachricht parsen
  PERFORM tedifact.segments__parse( _edi_id, _separators );

END $$ LANGUAGE plpgsql STRICT;
--


------------------------------------------------------
-- Einlesen einer peparsten EDI-Nachricht in PRODAT --
------------------------------------------------------


-- Liest ein bestimmtes Datenelement anhand seiner in der Tabelle t.edifact.translations hinterlegten Bedeutung aus.
CREATE OR REPLACE FUNCTION tedifact.dataelement__read( 
  _edi_id integer,                -- ID der EDI-Message
  _ecu_id integer,                -- ID des Kundenkunden
  _prodat_meaning varchar,        -- Bedeutung in PRODAt
  _edid_id_from integer = null,   -- Minimal-ID des geparsten Datenelements
  _edid_id_to integer = null      -- Maximal-ID des geparsten Datenelements
  ) RETURNS tedifact.edi_parsed_dataelements_short AS $$
  
  -- Sammelt zunächst alle Datenelemente, die zu den Eingabeparametern passen, ohne auf die Bedeutung zu achten
  WITH dataelements AS (        
    SELECT *
    FROM tedifact.edi_parsed_dataelements
    JOIN tedifact.edi_parsed_subsegments ON psubs_id = pdata_psubs_id
    JOIN tedifact.edi_incoming ON inc_id = _edi_id
    JOIN tedifact.customers ON cust_id = _ecu_id
    JOIN tedifact.edi_translation ON trans_prodat_meaning = _prodat_meaning AND trans_customer = cust_id  
    JOIN tedifact.dataelements ON data_id = pdata_data_id
    JOIN tedifact.subsegments ON subs_id = data_subs_id
    WHERE 
      pdata_inc_id = _edi_id
    AND trans_segment = subs_segment 
        AND ( _edid_id_from IS null OR _edid_id_from <= pdata_id )
        AND ( _edid_id_to   IS null OR _edid_id_to   >= pdata_id )   
  ),
  -- Filtert die oben gesammelten Datenelemente nach denen, die der übermittelten Bedeutung in PRODAT ensprechen
  translated_dataelements AS (
    SELECT psubs_psegm_id, pdata_id, pdata_content, trans_condition_data_code
    FROM dataelements
    WHERE trans_subs_pos = subs_pos 
      AND trans_data_code = data_code
  ),
  -- Sammelt die passenden Datenelemente, welche in der Bedingung in tedifact.translations angegeben sind.
  conditionelement AS (
    SELECT psubs_psegm_id
    FROM dataelements
    WHERE 
          trans_condition_subs_pos = subs_pos 
      AND trans_condition_data_code = data_code 
      AND trans_condition_value = pdata_content 
  )  
  -- Filtert nochmals die Datenelemente, die der angegebenne Bedingung entsprechen (sofern Bedungung vorhanden).
  SELECT ( pdata_id, pdata_content, _prodat_meaning ) FROM (
    SELECT pdata_id, pdata_content, _prodat_meaning 
    FROM translated_dataelements
    LEFT JOIN conditionelement ON conditionelement.psubs_psegm_id = translated_dataelements.psubs_psegm_id
    WHERE translated_dataelements.trans_condition_data_code IS null OR conditionelement.psubs_psegm_id IS NOT null

    UNION 

    -- Wurde nichts gefunden, dann Dummyrückgabe zwecks Fehlermeldung.
    SELECT null::integer, null::varchar, _prodat_meaning
  ) AS x
  ORDER BY x.pdata_id NULLS LAST
  LIMIT 1;
  
$$ LANGUAGE sql STABLE; 
--


-- Prüft, ob ein Datenelement tatsächlich vorhanden ist.
CREATE OR REPLACE FUNCTION tedifact.dataelement__to__string( _de tedifact.edi_parsed_dataelements_short ) RETURNS varchar AS $$
BEGIN

  IF coalesce( _de.pdata_content, '' ) = '' THEN
    RAISE EXCEPTION 'Element % ist leer', _de.meaning;
  END IF;
  
  RETURN _de.pdata_content;

END $$ language plpgsql;
--

-- Prüft, ob ein Datenelement tatsächlich vorhanden ist und eine ganze Zahl darstellt.
CREATE OR REPLACE FUNCTION tedifact.dataelement__to__integer( _de tedifact.edi_parsed_dataelements_short ) RETURNS integer AS $$
DECLARE
  _s varchar;
BEGIN

  _s := tedifact.dataelement__to__string( _de );

  IF NOT isInteger( _s ) THEN
    RAISE EXCEPTION 'Element % ist kein Integer: %', _de.meaning, _s;
  END IF;
  
  RETURN _s::integer;

END $$ language plpgsql;
--

-- Prüft, ob ein Datenelement tatsächlich vorhanden ist und eine Fließkommazahl darstellt.
CREATE OR REPLACE FUNCTION tedifact.dataelement__to__numeric( _de tedifact.edi_parsed_dataelements_short ) RETURNS numeric AS $$
DECLARE
  _s varchar;
BEGIN

  _s := tedifact.dataelement__to__string( _de );

  IF NOT isNumeric( _s ) THEN
    RAISE EXCEPTION 'Element % ist keine Zahl: %', _de.meaning, _s;
  END IF;
  
  RETURN _s::numeric;

END $$ language plpgsql;
--

-- Prüft, ob ein Datenelement tatsächlich vorhanden ist und ein Datum darstellt.
CREATE OR REPLACE FUNCTION tedifact.dataelement__to__date( _de tedifact.edi_parsed_dataelements_short ) RETURNS date AS $$
DECLARE
  _s varchar;
BEGIN

  _s := tedifact.dataelement__to__string( _de );

  IF NOT isDate( _s ) THEN
    RAISE EXCEPTION 'Element % ist kein Datum: %', _de.meaning, _s;
  END IF;
  
  RETURN _s::date;

END $$ language plpgsql;
--


-- Liest eine geparste EDI-Nachricht in PRODAT ein
CREATE OR REPLACE FUNCTION tedifact.ldsdok__ordrsp__get( _edi_id integer ) RETURNS varchar AS $$
DECLARE
  _content varchar;
  _a2_krz varchar;
  _ecu_id integer;
  _aknr varchar;
  _aknr_edi varchar;
  _ak_bez_edi varchar;
  _mce integer;
  _bestelldatum date;
  _steuersatz numeric;
  _waehrung varchar;
  _fremdbestellnummer varchar;
  _count integer;
  _p tedifact.edi_parsed_dataelements_short;
  _pos_index1 integer;
  _pos_index2 integer;
  _pos integer;
  _menge numeric;
  _preis numeric;
  _lieferdatum date;
  _ld_auftg varchar;
  _ld_konto integer;
  _rabatt_edi tedifact.edi_parsed_dataelements_short;
  _rabatt_faktor numeric;
  _rabatt_prozente numeric;
BEGIN

  -- Kunde und Nachrichentyp müssen stimmen
  SELECT inc_content, cust_a2_krz, cust_id INTO _content, _a2_krz, _ecu_id
  FROM tedifact.edi_incoming
  LEFT JOIN tedifact.customers 
    ON cust_customer_cimpcs = tsystem.settings__get('KUNDE') 
  AND cust_customer_external = inc_vendor
  AND cust_message_type = inc_message_type
  WHERE _edi_id = inc_id AND inc_message_type = 'ORDRSP';
  
  -- Lieferant sollte in PRODAT bekannt sein
  IF _a2_krz IS null THEN  
    RAISE EXCEPTION 'Unbekannter Lieferant % % %', TSystem.settings__get('KUNDE'), _a2_krz, 'ORDRSP';  
  END IF;
  
  -- Nicht positionsbezogene Daten ermitteln
  _bestelldatum := tedifact.dataelement__to__date( tedifact.dataelement__read( _edi_id, _ecu_id, 'Bestelldatum' ));
  _steuersatz := tedifact.dataelement__to__numeric( tedifact.dataelement__read( _edi_id, _ecu_id, 'Steuersatz' ));
  _waehrung := tedifact.dataelement__to__string( tedifact.dataelement__read( _edi_id, _ecu_id, 'Währung' ));
  _fremdbestellnummer := tedifact.dataelement__to__string( tedifact.dataelement__read( _edi_id, _ecu_id, 'Fremdbestellnummer' ));
  
  -- Erste Bestellposition auslesen
  _p := tedifact.dataelement__read( _edi_id, _ecu_id, 'Bestellpositionsnummer' );
  _pos_index1 := _p.pdata_id;
  IF NOT isInteger( _p.pdata_content ) THEN
    RAISE EXCEPTION 'Ungültige Positionsnummer: %', _p.pdata_content;
  END IF;
  _pos := _p.pdata_content::int;
    
  _count := 1;

  -- Absicherung gegen Endloschleifen, mehr als 100 Bestellpositionen wird es wohl nicht geben.
  WHILE _count <= 100 LOOP
  
    -- Bestimmung der nächsten Bestellposition, denn es werden im Folgenden nur Daten gesucht, die zu dieser Position gehören.
    _p := tedifact.dataelement__read( _edi_id, _ecu_id, 'Bestellpositionsnummer', _pos_index1 + 1 );
    _pos_index2 := _p.pdata_id;
  
    -- Artikel in PRODAT suchen
    _aknr_edi := tedifact.dataelement__to__string( tedifact.dataelement__read( _edi_id, _ecu_id, 'Artikelnummer', _pos_index1, _pos_index2 ));
    _ak_bez_edi := tedifact.dataelement__to__string( tedifact.dataelement__read( _edi_id, _ecu_id, 'Artikelbezeichnung', _pos_index1, _pos_index2 ));
    _aknr := ak_nr FROM art WHERE ak_nr = _aknr_edi;
    IF _aknr IS null THEN
    -- Wenn Artikel nicht gefunden, dann neu anlegen mit AC 5000 (Werkzeuge).
    INSERT INTO art
      ( ak_nr, ak_ac, ak_bez, ak_standard_mgc )
    VALUES
      ( _aknr_edi, 5000, _ak_bez_edi, 1 );

    _aknr := _aknr_edi;
    END IF;
    
    -- Mengencode für "Stück" auslesen
    _mce := m_id FROM artmgc WHERE m_ak_nr = _aknr AND m_mgcode = 1;
    IF _mce IS null THEN
      RAISE EXCEPTION 'Kein Mengencode für "Stück": Artikelnummer %', _aknr;
    END IF; 

    
    _menge := tedifact.dataelement__to__numeric( tedifact.dataelement__read( _edi_id, _ecu_id, 'Bestellmenge', _pos_index1, _pos_index2 ));
    _preis := tedifact.dataelement__to__numeric( tedifact.dataelement__read( _edi_id, _ecu_id, 'Positionspreis (Netto)', _pos_index1, _pos_index2 ));
    _lieferdatum := tedifact.dataelement__to__date( tedifact.dataelement__read( _edi_id, _ecu_id, 'Lieferdatum', _pos_index1, _pos_index2 ));
    
    -- Erlöskonto "Werkzeuge"
    _ld_konto := 4985;
  
    -- Position bereits in PRODAT vorahangen? Dann übergehe sie.
    IF EXISTS( SELECT 1 FROM ldsdok WHERE ld_kn = _a2_krz AND ld_post6 = _fremdbestellnummer AND ld_pos = _pos ) THEN
      CONTINUE;
    END IF;

    -- Auftragsnummer erst nach Duplikatsprüfung ermitteln, zur Vermeidung von Leerstellen
    IF _ld_auftg IS null THEN
      _ld_auftg := nummer FROM teinkauf.generate_bestellnummer_ext_by_agnr( 'E' );
    END IF;
  
    -- Rabatt
    _rabatt_edi := tedifact.dataelement__read( _edi_id, _ecu_id, 'Rabatt +/-', _pos_index1, _pos_index2 );
    _rabatt_prozente := 0;
    IF _rabatt_edi IS NOT null THEN
      -- A = Allowance = Rabatt, C = Charge = Aufschlag
      CASE _rabatt_edi.pdata_content 
        WHEN 'A' THEN _rabatt_faktor := 1;
        WHEN 'C' THEN _rabatt_faktor := -1;
        ELSE _rabatt_faktor := 0; 
      END CASE;
      
      _rabatt_prozente := tedifact.dataelement__to__numeric( tedifact.dataelement__read( _edi_id, _ecu_id, 'Rabatt in Prozent', _rabatt_edi.pdata_id, _pos_index2 ));
      _rabatt_prozente := _rabatt_prozente * _rabatt_faktor;
    END IF;

    -- todo: Rechnungs- und Lieferadresse
  
    -- Einfügen der Bestellposition
    -- Menge und Rabatt sind herauszurechnen
   _preis := _preis / _menge / coalesce((1 - _rabatt_prozente / 100), 1);
    INSERT INTO ldsdok
      ( ld_code, ld_auftg, ld_pos, ld_kn, ld_krzl, ld_krzf, ld_aknr, ld_stk, ld_abnr, ld_abdat,
        ld_term, ld_terml, ld_datum, ld_ep, ld_steuproz, ld_mce, ld_waer, ld_post6, ld_post7, ld_konto,
        ld_ekref, insert_date, insert_by, modified_date, modified_by, ld_arab
      )
    VALUES
      ( 'E', _ld_auftg, _pos, _a2_krz, _a2_krz, _a2_krz, _aknr, _menge, _fremdbestellnummer, _bestelldatum,
        _lieferdatum, _lieferdatum, _bestelldatum, _preis, _steuersatz, _mce, _waehrung, _bestelldatum, null, _ld_konto,
        null, now(), 'CIMPCS', now(), 'CIMPCS', _rabatt_prozente
      );
    
    -- Keine nächste Bestellposition? Dann war es das.
    EXIT WHEN ( _pos_index2 IS null );
    _count := _count + 1;

    -- Die "nächste Position" wird zur "aktuellen Position".
    _pos := _p.pdata_content::int;
    _pos_index1 := _pos_index2;
  END LOOP;

  RETURN _ld_auftg;

END $$ LANGUAGE plpgsql STRICT;
--


-- Parsen einer EDI-Message und anschließendem Import nach PRODAT.
-- Dies ist die einzige Funktion, doe von außerhalb des Schemas tedifact aufgerufen werden sollte.
CREATE OR REPLACE FUNCTION tedifact.ldsdok__ordrsp__create( _message_id varchar ) RETURNS varchar AS $$
DECLARE
  _edi_id integer;
  _err_msg varchar;
  _ld_auftg varchar;
BEGIN

  -- Primärschlüssel der EDI-Nachricht anhand der übergebenen Message-ID ermitteln
  _edi_id := inc_id FROM tedifact.edi_incoming WHERE inc_origin = _message_id AND inc_success IS null;
  
  _err_msg := null; _ld_auftg := null;

  -- Parsen und Daten nach PRODAT übertragen, Exceptions werden weggefangen
  BEGIN
    PERFORM tedifact.message__parse( _message_id  );
    _ld_auftg := tedifact.ldsdok__ordrsp__get( _edi_id );
  EXCEPTION WHEN others THEN
    _err_msg := sqlerrm;
  END;

  -- EDI-Nachricht im Erfolgsfall mit der Bestellnummer versehen
  -- Falls eine Exception kam, wird diese ebenfalls hier hinterlegt.
  UPDATE tedifact.edi_incoming SET inc_processing = now(), inc_error_message = _err_msg, inc_success = _ld_auftg WHERE inc_id = _edi_id;

  -- Auftragsnummr zurüch, steht hier null, dann gab es ein Problem.
  RETURN _ld_auftg;  

END $$ language plpgsql STRICT;
--
